//				Package : omnithread
// omnithread/nt.h		Created : 6/95 tjr
//
//    Copyright (C) 2002-2025 Apasphere Ltd
//    Copyright (C) 1995-1997 Olivetti & Oracle Research Laboratory
//
//    This file is part of the omnithread library
//
//    The omnithread library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library. If not, see http://www.gnu.org/licenses/
//
//
// OMNI thread implementation classes for NT threads.
//

#ifndef __omnithread_nt_h_
#define __omnithread_nt_h_

#ifndef WIN32_LEAN_AND_MEAN
#  define WIN32_LEAN_AND_MEAN
#  define OMNI_DEFINED_WIN32_LEAN_AND_MEAN
#endif

// TryEnterCriticalSection was introduced in Windows NT 4.0, so we need to
// tell the Windows header files to target that version (or later)
#ifndef WINVER
#  define WINVER 0x0400
#endif
#ifndef _WIN32_WINNT
#  define _WIN32_WINNT WINVER
#endif

#include <windows.h>

#ifdef OMNI_DEFINED_WIN32_LEAN_AND_MEAN
#  undef WIN32_LEAN_AND_MEAN
#  undef OMNI_DEFINED_WIN32_LEAN_AND_MEAN
#endif


#ifndef __BCPLUSPLUS__
#define OMNI_THREAD_WRAPPER \
    unsigned __stdcall omni_thread_wrapper(LPVOID ptr)
#else
#define OMNI_THREAD_WRAPPER \
    void _USERENTRY omni_thread_wrapper(void *ptr)
#endif

extern "C" OMNI_THREAD_WRAPPER;



///////////////////////////////////////////////////////////////////////////
//
// Mutex
//
///////////////////////////////////////////////////////////////////////////

class _OMNITHREAD_NTDLL_ omni_mutex {
public:
  omni_mutex();
  ~omni_mutex();

  inline void lock()    { EnterCriticalSection(&crit); }
  inline void unlock()  { LeaveCriticalSection(&crit); }

  inline int  trylock() { return TryEnterCriticalSection(&crit); }
  // if mutex is unlocked then lock it and return 1 (true).
  // If it is already locked then return 0 (false).

  friend class omni_condition;

private:
  // dummy copy constructor and operator= to prevent copying
  omni_mutex(const omni_mutex&);
  omni_mutex& operator=(const omni_mutex&);

  CRITICAL_SECTION crit;
};



///////////////////////////////////////////////////////////////////////////
//
// Condition variable
//
///////////////////////////////////////////////////////////////////////////

class _OMNITHREAD_NTDLL_ omni_condition {
public:
  omni_condition(omni_mutex* m);
  // constructor must be given a pointer to an existing mutex. The
  // condition variable is then linked to the mutex, so that there is an
  // implicit unlock and lock around wait() and timed_wait().

  ~omni_condition();

  void wait();
  // wait for the condition variable to be signalled.  The mutex is
  // implicitly released before waiting and locked again after waking up.
  // If wait() is called by multiple threads, a signal may wake up more
  // than one thread.  See POSIX threads documentation for details.

  int timedwait(omni_time_t::secs_t secs, omni_time_t::ns_t nanosecs = 0);
  // timedwait() is given an absolute time to wait until.  To wait for a
  // relative time from now, use omni_thread::get_time. See POSIX threads
  // documentation for why absolute times are better than relative.
  // Returns 1 (true) if successfully signalled, 0 (false) if time
  // expired.

  inline int timedwait(const omni_time_t& t)
  {
    return timedwait(t.s, t.ns);
  }

  void signal();
  // if one or more threads have called wait(), signal wakes up at least
  // one of them, possibly more.  See POSIX threads documentation for
  // details.

  void broadcast();
  // broadcast is like signal but wakes all threads which have called
  // wait().

private:
  // dummy copy constructor and operator= to prevent copying
  omni_condition(const omni_condition&);
  omni_condition& operator=(const omni_condition&);

  omni_mutex*        mutex;
  CONDITION_VARIABLE cond;
};


///////////////////////////////////////////////////////////////////////////
//
// Counting semaphore
//
///////////////////////////////////////////////////////////////////////////

// We only use native Windows semaphores if we are not using atomic mutex /
// condition.

#define OMNI_HAVE_PLATFORM_SEMAPHORE

class _OMNITHREAD_NTDLL_ omni_semaphore {
public:
  omni_semaphore(unsigned int initial = 1);
  ~omni_semaphore();

  void wait();
  // if semaphore value is > 0 then decrement it and carry on. If it's
  // already 0 then block.

  int trywait();
  // if semaphore value is > 0 then decrement it and return 1 (true).
  // If it's already 0 then return 0 (false).

  void post();
  // if any threads are blocked in wait(), wake one of them up. Otherwise
  // increment the value of the semaphore.

private:
  // dummy copy constructor and operator= to prevent copying
  omni_semaphore(const omni_semaphore&);
  omni_semaphore& operator=(const omni_semaphore&);

  HANDLE nt_sem;
};


#define OMNI_THREAD_IMPLEMENTATION			\
    HANDLE handle;					\
    DWORD  nt_id;					\
    void*  return_val;					\
    HANDLE cond_semaphore;				\
    omni_thread* cond_next;				\
    omni_thread* cond_prev;				\
    BOOL cond_waiting;					\
    static int nt_priority(priority_t);			\
    friend class omni_condition;			\
    friend OMNI_THREAD_WRAPPER;

#endif
