// -*- Mode: C++; -*-
//				Package : omnithread
// omnithread/atomic.cc		Created : 2023/09/25 dgrisby
//
//    Copyright (C) 2023-2024 Apasphere Ltd
//
//    This file is part of the omnithread library
//
//    The omnithread library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library. If not, see http://www.gnu.org/licenses/
//

//
// Atomic operations for platforms that support them.

#include <omnithread.h>

#ifdef OMNI_ATOMIC_MUTEX

namespace {

  // Number of times to spin.
  static constexpr unsigned int SPIN_MAX = 40;
  
  struct queue_item {
    inline queue_item()
      : next(nullptr), prev(nullptr), tail(nullptr),
        cond(&mutex), should_sleep(false), signalled(false) {}

    // (Doubly) linked list. The top queue item also maintains a pointer to
    // the tail of the queue. prev pointers are only used in condition
    // variables. Mutexes use a single linked list.

    queue_item* next;
    queue_item* prev;
    queue_item* tail;

    omni_platform_mutex     mutex;
    omni_platform_condition cond;
    bool                    should_sleep; // protected by mutex
    bool                    signalled;    // protected by queue lock

    inline void sleep()
    {
      omni_mutex_lock l(mutex);
      while (should_sleep)
        cond.wait();
    }

    inline bool timed_sleep(const omni_time_t& deadline)
    {
      omni_mutex_lock l(mutex);
      while (should_sleep) {
        if (!cond.timedwait(deadline))
          return false;
      }
      return true;
    }

    inline void wake()
    {
      omni_mutex_lock l(mutex);
      should_sleep = false;
      cond.signal();
    }
  };
  
};

///////////////////////////////////////////////////////////////////////////
//
// Mutex
//
///////////////////////////////////////////////////////////////////////////

void omni_mutex::lock_slow()
{
  unsigned int spins = 0;

  while (1) {
    uintptr_t current = value_.load();

    if (!(current & LOCKED_BIT)) {
      // Not currently locked

      if (value_.compare_exchange_weak(current, current | LOCKED_BIT)) {
        // We locked it.
        return;
      }
    }

    // If there is no queue, spin a number of times.

    if (!(current & ~BIT_MASK) && spins < SPIN_MAX) {
      ++spins;
      omni_thread::yield();
      continue;
    }

    // We need to be queued. Try to acquire the queue lock.

    queue_item self; // Used as a single linked list -- prev not used

    current = value_.load();

    if ((current & QUEUE_LOCKED_BIT) || !(current & LOCKED_BIT) ||
        !value_.compare_exchange_weak(current, current | QUEUE_LOCKED_BIT)) {

      // Queue locked by another thread, or the mutex was unlocked.
      omni_thread::yield();
      continue;
    }

    self.should_sleep = true;

    // Add ourselves to the queue.

    queue_item* head = reinterpret_cast<queue_item*>(current & ~BIT_MASK);
    if (head) {
      // There is an existing queue -- add ourselves to the tail.
      head->tail->next = &self;
      head->tail       = &self;

      // Unlock the queue lock.
      value_.store(current & ~QUEUE_LOCKED_BIT);
    }
    else {
      // No existing queue. We are the head of the new queue.
      head      = &self;
      self.tail = &self;

      // Install ourselves as head of the queue and unlock the queue lock.
      // The mutex is still guaranteed to be locked by another thread,
      // because it cannot be unlocked without getting the queue lock.

      value_.store((reinterpret_cast<uintptr_t>(head)) | LOCKED_BIT);
    }

    // Wait to be woken up by a thread releasing the mutex. Another thread
    // may have already chosen to wake us before we get here.
    self.sleep();

    // Loop to try to acquire the mutex.
  }
}


void omni_mutex::unlock_slow()
{
  uintptr_t current;
  
  while (1) {
    current = value_.load();

    if (current == LOCKED_BIT) {
      if (value_.compare_exchange_weak(current, 0)) {
        // Nobody queued and successfully unlocked.
        return;
      }
      // Compare-exchange has failed. Go around again.
      omni_thread::yield();
      continue;
    }

    if (current & QUEUE_LOCKED_BIT) {
      // Queue is locked -- another thread must be adding itself to the queue.
      omni_thread::yield();
      continue;
    }

    // Lock the queue.
    if (value_.compare_exchange_weak(current, current | QUEUE_LOCKED_BIT))
      break;

    // Failed to lock the queue -- try again.
    omni_thread::yield();
  }
        
  current = value_.load();

  queue_item* head     = reinterpret_cast<queue_item*>(current & ~BIT_MASK);
  queue_item* new_head = head->next;

  if (new_head)
    new_head->tail = head->tail;

  // Store the new queue head, unlocking the queue and the mutex.

  value_.store(reinterpret_cast<uintptr_t>(new_head));

  head->next = nullptr;
  head->tail = nullptr;

  // Wake up the thread that was the head. It will probably then acquire the
  // mutex, but if it races with a new incoming thread, it will queue itself
  // again.

  head->wake();
}


///////////////////////////////////////////////////////////////////////////
//
// Condition variable
//
///////////////////////////////////////////////////////////////////////////

inline uintptr_t omni_condition::acquire_queue_lock()
{
  // Acquire the queue lock.

  uintptr_t current;
    
  while (1) {
    current = value_.load();

    if ((current & QUEUE_LOCKED_BIT) ||
        !value_.compare_exchange_weak(current, current | QUEUE_LOCKED_BIT)) {

      // Locked by another thread. Spin.
      omni_thread::yield();
    }
    else {
      // We now hold the queue lock.
      break;
    }
  }
  return current;
}


void omni_condition::wait()
{
  queue_item self;
  self.should_sleep = true;

  uintptr_t   current = acquire_queue_lock();
  queue_item* head    = reinterpret_cast<queue_item*>(current & ~BIT_MASK);

  if (!head) {
    // No existing queue. We are the head of the new queue.
    head      = &self;
    self.tail = &self;

    // Install ourselves as head of the queue and unlock the queue lock.
    value_.store((reinterpret_cast<uintptr_t>(head)));
  }
  else {
    // There is an existing queue -- add ourselves to the tail.
    head->tail->next = &self;
    self.prev        = head->tail;
    head->tail       = &self;

    // Unlock the queue lock.
    value_.store(current & ~QUEUE_LOCKED_BIT);
  }

  // Unlock the linked omni_mutex.
  mutex_->unlock();

  // Wait to be signalled.
  self.sleep();

  // Re-lock the linked omni_mutex.
  mutex_->lock();
}


bool omni_condition::timedwait(const omni_time_t& deadline)
{
  queue_item self;
  self.should_sleep = true;

  uintptr_t   current = acquire_queue_lock();
  queue_item* head    = reinterpret_cast<queue_item*>(current & ~BIT_MASK);

  if (!head) {
    // No existing queue. We are the head of the new queue.
    head      = &self;
    self.tail = &self;

    // Install ourselves as head of the queue and unlock the queue lock.
    value_.store((reinterpret_cast<uintptr_t>(head)));
  }
  else {
    // There is an existing queue -- add ourselves to the tail.
    head->tail->next = &self;
    self.prev        = head->tail;
    head->tail       = &self;

    // Unlock the queue lock.
    value_.store(current & ~QUEUE_LOCKED_BIT);
  }

  // Unlock the linked omni_mutex.
  mutex_->unlock();

  // Wait to be signalled.
  bool signalled = self.timed_sleep(deadline);

  if (!signalled) {
    // Timed out. Reacquire the queue lock.

    current = acquire_queue_lock();

    if (!self.signalled) {
      // Timed out. Remove ourselves from the queue.

      if (!self.prev) {
        // Most likely: we are head of the queue (in fact probably the only
        // queue member).
        queue_item* new_head = self.next;

        if (new_head) {
          new_head->prev = nullptr;
          new_head->tail = self.tail;
        }

        // Store the new queue head (or empty) and unlock the queue.
        value_.store(reinterpret_cast<uintptr_t>(new_head));
      }
      else {
        // Not the head of the queue
        queue_item* head = reinterpret_cast<queue_item*>(current & ~BIT_MASK);
        
        self.prev->next = self.next;
        self.next->prev = self.prev;

        if (head->tail == &self) {
          // We were at the tail of the queue
          head->tail = self.prev;
        }

        // Store the existing queue head and unlock the queue.
        value_.store(reinterpret_cast<uintptr_t>(head));
      }
    }
    else {
      // The platform condition timedwait timed out, but a thread raced to
      // signal us. It has removed us from the queue, but it might not yet
      // have called wake(). We must wait until it defintely has done before
      // returning, otherwise self will have gone too early.

      // Unlock the queue.
      value_.store(current & ~QUEUE_LOCKED_BIT);

      // Make sure the signalling thread has had a chance to call wake().
      self.sleep();
      
      signalled = true;
    }      
  }

  // Re-lock the linked omni_mutex.
  mutex_->lock();

  return signalled;
}


void omni_condition::do_signal()
{
  uintptr_t   current  = acquire_queue_lock();
  queue_item* head     = reinterpret_cast<queue_item*>(current & ~BIT_MASK);
  queue_item* new_head = nullptr;

  if (head) {
    // We will wake the top thread from the queue.
    new_head = head->next;

    if (new_head) {
      new_head->prev = nullptr;
      new_head->tail = head->tail;
    }
    head->signalled = true;
  }

  // Store the new queue head (or empty) and unlock the queue.
  value_.store(reinterpret_cast<uintptr_t>(new_head));

  if (head)
    head->wake();
}


void omni_condition::do_broadcast()
{
  uintptr_t   current  = acquire_queue_lock();
  queue_item* head     = reinterpret_cast<queue_item*>(current & ~BIT_MASK);

  // While holding the queue lock, mark all the threads as signalled

  queue_item* item = head;
  while (item) {
    item->signalled = true;
    item = item->next;
  }

  // Store empty queue and unlock it.
  value_.store(0);

  // Wake all the threads.

  item = head;
  while (item) {
    item->wake();
    item = item->next;
  }  
}


#endif // OMNI_ATOMIC_MUTEX
